document.addEventListener("DOMContentLoaded", function () {
    function formatDate(element, attr, prefix, type) {
        var value = element.getAttribute(attr);
        var date = new Date(value);
        var now = new Date();

        // Format the date as YYYY-MM-DD
        var year = date.getFullYear();
        var month = (date.getMonth() + 1).toString().padStart(2, '0');
        var day = date.getDate().toString().padStart(2, '0');
        var text = year + "-" + month + "-" + day;

        // Add prefix if provided
        if (prefix) {
            text = prefix + text;
        }

        // Keep the CSS classes for styling
        if (type === "due") {
            var today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
            var dueDate = new Date(date.getFullYear(), date.getMonth(), date.getDate());
            var daysDiff = Math.round((dueDate - today) / 86400000);

            // Clear existing classes
            element.classList.remove("today", "tomorrow", "yesterday", "overdue", "future");

            // Add appropriate class based on relative date
            if (daysDiff === 0) {
                element.classList.add("today");
            } else if (daysDiff === 1) {
                element.classList.add("tomorrow");
            } else if (daysDiff === -1) {
                element.classList.add("yesterday");
            } else if (daysDiff < 0) {
                element.classList.add("overdue");
            } else {
                element.classList.add("future");
            }
        }

        element.textContent = text;
        return date;
    }

    function createGroupHeader(text, color) {
        var div = document.createElement("div");
        div.className = "group-header";
        div.style.color = color;
        div.textContent = text;
        return div
    }

    document.querySelectorAll(".task-card").forEach(function (card) {
        var taskList = card.querySelector(".task-list");
        var moreIcon = card.querySelector(".more-icon-container");
        var optionsMenu = card.querySelector(".options-menu");
        var sortOptions = card.querySelectorAll(".sort-option");
        var tasks = [];
        var sortType = "my-order";
        card.querySelectorAll("[data-due]").forEach(function (el) {
            formatDate(el, "data-due", "", "due")
        });
        card.querySelectorAll("[data-completed]").forEach(function (el) {
            formatDate(el, "data-completed", "Completed: ", "completed")
        });
        card.querySelectorAll(".task:not(.completed_tasks .task)").forEach(function (task, index) {
            var titleEl = task.querySelector(".task-title");
            var dueEl = task.querySelector(".due");
            tasks.push({
                element: task,
                title: titleEl ? titleEl.textContent : "",
                due: dueEl ? dueEl.getAttribute("data-due") : null,
                originalIndex: index
            })
        });
        var completedCount = card.querySelectorAll(".completed_tasks .task").length;
        var summaryText = card.querySelector(".completed_tasks .summary-text");
        if (summaryText) summaryText.textContent = `Completed (${completedCount})`;

        function sortTasks(sort) {
            sortType = sort;
            taskList.classList.remove("sort-my-order", "sort-date", "sort-title");
            taskList.classList.add("sort-" + sort);
            var sortedTasks = [...tasks];
            if (sort === "my-order") {
                sortedTasks.sort(function (a, b) {
                    return a.originalIndex - b.originalIndex
                })
            } else if (sort === "date") {
                var grouped = {past: [], today: [], tomorrow: [], future: {}, noDate: []};
                var now = new Date();
                var today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
                var tomorrow = new Date(today);
                tomorrow.setDate(tomorrow.getDate() + 1);
                sortedTasks.forEach(function (task) {
                    if (task.due) {
                        var due = new Date(task.due);
                        var dueDate = new Date(due.getFullYear(), due.getMonth(), due.getDate());
                        if (dueDate < today) grouped.past.push(task); else if (dueDate.getTime() === today.getTime()) grouped.today.push(task); else if (dueDate.getTime() === tomorrow.getTime()) grouped.tomorrow.push(task); else {
                            var key = due.toDateString();
                            grouped.future[key] = grouped.future[key] || [];
                            grouped.future[key].push(task)
                        }
                    } else {
                        grouped.noDate.push(task)
                    }
                });
                grouped.past.sort(function (a, b) {
                    return new Date(a.due) - new Date(b.due)
                });
                grouped.today.sort(function (a, b) {
                    return new Date(a.due) - new Date(b.due)
                });
                grouped.tomorrow.sort(function (a, b) {
                    return new Date(a.due) - new Date(b.due)
                });
                grouped.noDate.sort(function (a, b) {
                    return a.originalIndex - b.originalIndex
                });
                sortedTasks = grouped
            } else if (sort === "title") {
                sortedTasks.sort(function (a, b) {
                    return a.title.localeCompare(b.title)
                })
            }
            var completedTasks = taskList.querySelector(".completed_tasks");
            taskList.querySelectorAll(".task, .group-header, .no-tasks").forEach(function (el) {
                if (!el.closest(".completed_tasks")) el.remove();
            });
            if (tasks.length === 0 && !completedTasks) {
                var noTasks = document.createElement("div");
                noTasks.className = "no-tasks";
                noTasks.innerHTML = '<span class="material-icons no-tasks-icon">list</span><p>No tasks yet</p>';
                taskList.appendChild(noTasks)
            } else if (sort === "date") {
                if (grouped.past.length) {
                    taskList.insertBefore(createGroupHeader("Past", "#d93025"), completedTasks);
                    grouped.past.forEach(function (task) {
                        taskList.insertBefore(task.element, completedTasks)
                    })
                }
                if (grouped.today.length) {
                    taskList.insertBefore(createGroupHeader("Today", "#1a73e8"), completedTasks);
                    grouped.today.forEach(function (task) {
                        var due = task.element.querySelector(".due");
                        if (due) due.style.display = "none";
                        taskList.insertBefore(task.element, completedTasks)
                    })
                }
                if (grouped.tomorrow.length) {
                    taskList.insertBefore(createGroupHeader("Tomorrow", "#5f6368"), completedTasks);
                    grouped.tomorrow.forEach(function (task) {
                        var due = task.element.querySelector(".due");
                        if (due) due.style.display = "none";
                        taskList.insertBefore(task.element, completedTasks)
                    })
                }
                var futureKeys = Object.keys(grouped.future).sort(function (a, b) {
                    return new Date(a) - new Date(b)
                });
                futureKeys.forEach(function (key) {
                    var date = new Date(key);
                    var year = now.getFullYear();
                    var label = date.getFullYear() !== year ? `${date.getDate()} ${new Intl.DateTimeFormat("en-US", {month: "long"}).format(date)} ${date.getFullYear()}` : `${new Intl.DateTimeFormat("en-US", {weekday: "short"}).format(date)} ${date.getDate()} ${new Intl.DateTimeFormat("en-US", {month: "short"}).format(date)}`;
                    taskList.insertBefore(createGroupHeader(label, "#5f6368"), completedTasks);
                    grouped.future[key].sort(function (a, b) {
                        return new Date(a.due) - new Date(b.due)
                    });
                    grouped.future[key].forEach(function (task) {
                        var due = task.element.querySelector(".due");
                        if (due) due.style.display = "none";
                        taskList.insertBefore(task.element, completedTasks)
                    })
                });
                if (grouped.noDate.length) {
                    taskList.insertBefore(createGroupHeader("No date", "#5f6368"), completedTasks);
                    grouped.noDate.forEach(function (task) {
                        taskList.insertBefore(task.element, completedTasks)
                    })
                }
            } else {
                sortedTasks.forEach(function (task) {
                    taskList.insertBefore(task.element, completedTasks)
                })
            }
            if (completedTasks) taskList.appendChild(completedTasks);
            taskList.querySelectorAll(".task.subtask:not(.completed_tasks .task.subtask)").forEach(function (el) {
                el.style.marginLeft = (sort === "date" || sort === "title") ? "0" : "20px"
            })
        }

        taskList.addEventListener("scroll", function () {
            if (taskList.scrollTop > 0) card.classList.add("task-card--scrolled"); else card.classList.remove("task-card--scrolled")
        });
        moreIcon.addEventListener("click", function (e) {
            e.stopPropagation();
            moreIcon.classList.toggle("active");
            optionsMenu.classList.toggle("active")
        });
        document.addEventListener("click", function (e) {
            if (!optionsMenu.contains(e.target) && !moreIcon.contains(e.target)) {
                moreIcon.classList.remove("active");
                optionsMenu.classList.remove("active")
            }
        });
        sortOptions.forEach(function (option) {
            option.addEventListener("click", function (e) {
                e.stopPropagation();
                sortOptions.forEach(function (opt) {
                    opt.classList.remove("selected");
                    opt.querySelector(".sort-icon-container").innerHTML = ""
                });
                option.classList.add("selected");
                option.querySelector(".sort-icon-container").innerHTML = "<span class='sort-icon material-icons'>check</span>";
                sortTasks(option.getAttribute("data-sort"));
                moreIcon.classList.remove("active");
                optionsMenu.classList.remove("active")
            })
        });
        card.querySelectorAll(".expandable").forEach(function (el) {
            el.addEventListener("click", function (e) {
                e.stopPropagation();
                var content = el.closest(".task-content");
                var title = content.querySelector(".task-title");
                var notes = content.querySelector(".task-notes");
                var expanded = el.getAttribute("data-expanded") === "true";
                if (expanded) {
                    if (title) {
                        title.classList.remove("expanded");
                        title.setAttribute("data-expanded", "false")
                    }
                    if (notes) {
                        notes.classList.remove("expanded");
                        notes.setAttribute("data-expanded", "false")
                    }
                } else {
                    if (title) {
                        title.classList.add("expanded");
                        title.setAttribute("data-expanded", "true")
                    }
                    if (notes) {
                        notes.classList.add("expanded");
                        notes.setAttribute("data-expanded", "true")
                    }
                }
            })
        });
        sortTasks("my-order")
    });
    const taskContainer = document.querySelector('.task-container');
    const taskHeaders = document.querySelectorAll('.task-header');
    const placeholder = document.createElement('div');
    placeholder.className = 'task-card placeholder';
    taskHeaders.forEach(header => {
        header.setAttribute('draggable', 'true');
        header.addEventListener('dragstart', (e) => {
            const card = header.closest('.task-card');
            e.dataTransfer.setData('text/plain', card.dataset.index || '0');
            card.classList.add('dragging');
            taskContainer.classList.add('dragging-active');
            setTimeout(() => {
                card.style.display = 'none';
                taskContainer.insertBefore(placeholder, card);
                placeholder.style.width = `${card.offsetWidth}px`
            }, 0)
        });
        header.addEventListener('dragend', () => {
            const card = header.closest('.task-card');
            card.classList.remove('dragging');
            taskContainer.classList.remove('dragging-active');
            card.style.display = '';
            if (placeholder.parentNode) {
                placeholder.remove()
            }
        })
    });
    taskContainer.addEventListener('dragover', (e) => {
        e.preventDefault();
        const draggingCard = document.querySelector('.dragging');
        if (!draggingCard) return;
        const afterElement = getDragAfterElement(taskContainer, e.clientX);
        if (afterElement == null) {
            taskContainer.appendChild(placeholder)
        } else {
            taskContainer.insertBefore(placeholder, afterElement)
        }
    });
    taskContainer.addEventListener('drop', (e) => {
        e.preventDefault();
        const draggingCard = document.querySelector('.dragging');
        if (!draggingCard) return;
        taskContainer.insertBefore(draggingCard, placeholder);
        placeholder.remove();
        draggingCard.style.display = '';
        draggingCard.classList.remove('dragging');
        const newOrder = Array.from(taskContainer.querySelectorAll('.task-card')).map((card, index) => {
            card.dataset.index = index;
            return card.querySelector('h1').textContent
        });
        console.log('New order:', newOrder)
    });

    function getDragAfterElement(container, x) {
        const draggableElements = [...container.querySelectorAll('.task-card:not(.dragging):not(.placeholder)')];
        return draggableElements.reduce((closest, child) => {
            const box = child.getBoundingClientRect();
            const offset = x - box.left - box.width / 2;
            if (offset < 0 && offset > closest.offset) {
                return {offset: offset, element: child}
            } else {
                return closest
            }
        }, {offset: Number.NEGATIVE_INFINITY}).element
    }
})